/******************************************************************************
 *                  Shanghai ChipON Micro-Electronic Co.,Ltd                  *
 ******************************************************************************
 *  $File Name$       : kf32a156_qei.c                   	                  *
 *  $Author$          : ChipON AE/FAE Group                                   *
 *  $Data$            : 2021-07-08                                            *
 *  $AutoSAR Version  : V1.0	                                              *
 *  $Description$     : This document provides functions related to 		  *
 *  					quadrature coded pulse circuits (QEI) , including:	  *
 *          			+ Quadrature coded pulse circuit(QEI) unitialization  *
 *          			  function    										  *
 *          			+ Quadrature coded pulse circuit(QEI) function 		  *
 *          			  configuration function							  *
 *          			+ Quadrature coded pulse circuit(QEI) interrupt 	  *
 *          			  management function								  *
 ******************************************************************************
 *  Copyright (C) by Shanghai ChipON Micro-Electronic Co.,Ltd                 *
 *  All rights reserved.                                                      *
 *                                                                            *
 *  This software is copyrght protected and proprietary to                    *
 *  Shanghai ChipON Micro-Electronic Co.,Ltd.                                 *
 ******************************************************************************
 *  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
 *                     		REVISON HISTORY                               	  *
 *  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
 *  Data       Version  Author        Description                             *
 *  ~~~~~~~~~~ ~~~~~~~~ ~~~~~~~~~~~~  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  *
 *  2021-07-08 00.01.00 FAE Group     new creat                               *
 *                                                                            *
 *                                                                            *
 *****************************************************************************/

/******************************************************************************
**                      		Include Files                                **
******************************************************************************/

#include "kf32a156_qei.h"
#include "kf32a156_rst.h"
#include "kf32a156_pclk.h"


/*******************************************************************************
**                    (QEI) Module private definition                         **
*******************************************************************************/

/* QEIx_CTL0Ĵʱܳʼ */
#define QEI_CTL0_TIMER_INIT_MASK        (QEI_CTL0_TXDIR \
                                       | QEI_CTL0_TXCS)

/* QEIx_CTL0Ĵʼ */
#define QEI_CTL0_INIT_MASK              (QEI_CTL0_INDRES \
                                       | QEI_CTL0_TXDIROUT)

/* QEIx_CTL1Ĵʼ */
#define QEI_CTL1_INIT_MASK              (QEI_CTL1_QEICLK \
                                       | QEI_CTL1_QEIOUT \
                                       | QEI_CTL1_QEICK \
                                       | QEI_CTL1_SWPAB)


/**
  *   ##### ·(QEI)ʼ #####
  */
/**
  *   λQEI裬ʹʱӡ
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *   ޡ
  */
void
QEI_Reset (QEI_SFRmap* QEIx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));

    if (QEIx == QEI0_SFR)
    {
        RST_CTL1_Peripheral_Reset_Enable(RST_CTL1_QEI0RST, TRUE);
        RST_CTL1_Peripheral_Reset_Enable(RST_CTL1_QEI0RST, FALSE);
        PCLK_CTL1_Peripheral_Clock_Enable(PCLK_CTL1_QEI0CLKEN, TRUE);
    }
    else if(QEIx == QEI1_SFR)
    {
        RST_CTL1_Peripheral_Reset_Enable(RST_CTL1_QEI1RST, TRUE);
        RST_CTL1_Peripheral_Reset_Enable(RST_CTL1_QEI1RST, FALSE);
        PCLK_CTL1_Peripheral_Clock_Enable(PCLK_CTL1_QEI1CLKEN, TRUE);
    }
    else
    {
        ;
    }
}

/**
  *   QEIģʼ
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *       qeiInitStruct: QEIģϢṹ塣
  *   ޡ
  */
void
QEI_Configuration(QEI_SFRmap* QEIx, QEI_InitTypeDef* qeiInitStruct)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(qeiInitStruct->m_IndexReset));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(qeiInitStruct->m_DirectionEn));
    CHECK_RESTRICTION(CHECK_QEI_SOURCE(qeiInitStruct->m_WorkClock));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(qeiInitStruct->m_DigitalFilterEn));
    CHECK_RESTRICTION(CHECK_QEI_DIGITAL_FILTER_CLK( \
                            qeiInitStruct->m_DigitalFilterPrescaler));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(qeiInitStruct->m_SwapABEn));

    /*---------------- Tx_CNTĴ ----------------*/
    QEIx->CNT = (uint32_t)qeiInitStruct->m_Counter;

    /*---------------- PPxڼĴ ---------------*/
    QEIx->PPX = (uint32_t)qeiInitStruct->m_Period;

    /*---------------- Tx_PRSCԤƵĴ ---------------*/
    QEIx->PRSC = (uint32_t)qeiInitStruct->m_Prescaler;

    /*---------------- QEIx_CTL1Ĵ ----------------*/
    /* ݽṹԱm_WorkClockQEICLKλ */
    /* ݽṹԱm_DigitalFilterEnQEIOUTλ */
    /* ݽṹԱm_DigitalFilterPrescalerQEICKλ */
    /* ݽṹԱm_SwapABEnSWPABλ */
    tmpreg = (qeiInitStruct->m_WorkClock) \
           | (qeiInitStruct->m_DigitalFilterEn << QEI_CTL1_QEIOUT_POS) \
           | (qeiInitStruct->m_DigitalFilterPrescaler) \
           | (qeiInitStruct->m_SwapABEn << QEI_CTL1_SWPAB_POS);
    QEIx->CTL1 = SFR_Config (QEIx->CTL1,
                          ~QEI_CTL1_INIT_MASK,
                          tmpreg);

    /*---------------- QEIx_CTL0Ĵ ----------------*/
    /* ݽṹԱm_IndexResetINDRESλ */
    /* ݽṹԱm_DirectionEnTXDIROUTλ */
    tmpreg = (qeiInitStruct->m_IndexReset << QEI_CTL0_INDRES_POS) \
           | (qeiInitStruct->m_DirectionEn << QEI_CTL0_TXDIROUT_POS);
    QEIx->CTL0 = SFR_Config (QEIx->CTL0,
                          ~QEI_CTL0_INIT_MASK,
                          tmpreg);
}

/**
  *   QEIģ鶨ʱʼ
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *       qeiTimerInitStruct: QEIʱ(T7/T8)Ϣṹָ롣
  *   ޡ
  */
void
QEI_Timer_Configuration(QEI_SFRmap* QEIx, QEI_TimerTypeDef* qeiTimerInitStruct)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));
    CHECK_RESTRICTION(CHECK_QEI_COUNT_DIRECT(qeiTimerInitStruct->m_CounterMode));
    CHECK_RESTRICTION(CHECK_QEI_TIMER_SOURCE(qeiTimerInitStruct->m_ClockSource));

    /*---------------- Tx_CNTĴ ----------------*/
    QEIx->CNT = (uint32_t)qeiTimerInitStruct->m_Counter;

    /*---------------- PPxڼĴ ---------------*/
    QEIx->PPX = (uint32_t)qeiTimerInitStruct->m_Period;

    /*---------------- TxԤƵĴ --------------*/
    QEIx->PRSC = (uint32_t)qeiTimerInitStruct->m_Prescaler;

    /*---------------- QEIx_CTL0Ĵ ----------------*/
    /* ݽṹԱm_CounterModeTXDIRλ */
    /* ݽṹԱm_ClockSourceTXCSλ */
    tmpreg = (qeiTimerInitStruct->m_CounterMode) \
           | (qeiTimerInitStruct->m_ClockSource);
    QEIx->CTL0 = SFR_Config (QEIx->CTL0,
                          ~QEI_CTL0_TIMER_INIT_MASK,
                          tmpreg);
}

/**
  *   ʼQEIģϢṹ塣
  *   qeiInitStruct: ָʼĽṹָ롣
  *   ޡ
  */
void
QEI_Struct_Init (QEI_InitTypeDef* qeiInitStruct)
{
    /* ʼ λüλʹ */
    qeiInitStruct->m_IndexReset = FALSE;
    /* ʼ λü״̬ʹ */
    qeiInitStruct->m_DirectionEn = FALSE;
    /* ʼ QEIʱ */
    qeiInitStruct->m_WorkClock = QEI_SOURCE_SCLK;
    /* ʼ QEI˲ʹ */
    qeiInitStruct->m_DigitalFilterEn = FALSE;
    /* ʼ QEI˲ʱӷƵ */
    qeiInitStruct->m_DigitalFilterPrescaler = QEI_DIGITAL_FILTER_CLK_DIV_1;
    /* ʼ AB뽻ʹ */
    qeiInitStruct->m_SwapABEn = FALSE;
    /* ʼ ʱֵ */
    qeiInitStruct->m_Counter = 0;
    /* ʼ ʱֵ */
    qeiInitStruct->m_Period = 0;
    /* ʼ ʱԤƵ */
    qeiInitStruct->m_Prescaler = 0;
}

/**
  *   ʼQEIʱϢṹ塣
  *   qeiTimerInitStruct: ָʼĽṹָ롣
  *   ޡ
  */
void
QEI_Timer_Struct_Init (QEI_TimerTypeDef* qeiTimerInitStruct)
{
    /* ʼ ʱֵ */
    qeiTimerInitStruct->m_Counter = 0;
    /* ʼ ʱֵ */
    qeiTimerInitStruct->m_Period = 0;
    /* ʼ ʱԤƵֵ */
    qeiTimerInitStruct->m_Prescaler = 0;
    /* ʼ  */
    qeiTimerInitStruct->m_CounterMode = QEI_COUNT_DIRECT_REVERSE;
    /* ʼ ʱʱԴѡ */
    qeiTimerInitStruct->m_ClockSource = QEI_TIMER_SOURCE_SCLK;
}
/**
  *   ##### ·(QEI)ʼ #####
  */


/**
  *   ##### ·(QEI)ú #####
  */
/**
  *   ӿģʽ
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFRQEI1_SFR
  *       PortMode: ӿģʽȡֵΧΪ:
  *                   QEI_ENCODER_OFF: ӿ/ʱر
  *                   QEI_GENERAL_PURPOSE_COUNTER: ʹܼ/ʱӿѡͨüģʽ
  *                   QEI_X2_INDEX_PULSE_RESET: ʹӿڣx2ģʽ帴λλü
  *                   QEI_X2_MATCH_RESET: ʹӿڣx2ģʽƥʱλλüPPX
  *                   QEI_X4_INDEX_PULSE_RESET: ʹӿڣx4ģʽ帴λλü
  *                   QEI_X4_MATCH_RESET: ʹӿڣx4ģʽƥʱλλüPPX
  *   ޡ
  */
void
QEI_Encoder_Mode_Config (QEI_SFRmap* QEIx, uint32_t PortMode)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));
    CHECK_RESTRICTION(CHECK_QEI_PORT_MODE(PortMode));

    /*---------------- QEI_CTL0ĴQEIMODλ ----------------*/
    tmpreg = PortMode << QEI_CTL0_QEIMOD0_POS;
    QEIx->CTL0 = SFR_Config (QEIx->CTL0, ~QEI_CTL0_QEIMOD, tmpreg);
}

/**
  *   λüλʹܡ
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *       NewState: λüλʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
QEI_Location_Counter_Reset_Enable (QEI_SFRmap* QEIx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- QEI_CTL0ĴINDRESλ ----------------*/
    if (NewState != FALSE)
    {
        /* 帴λλü */
        SFR_SET_BIT_ASM(QEIx->CTL0, QEI_CTL0_INDRES_POS);
    }
    else
    {
        /* 岻λλü */
        SFR_CLR_BIT_ASM(QEIx->CTL0, QEI_CTL0_INDRES_POS);
    }
}

/**
  *   QEIʱʱԴ
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *       TimerClock: QEIʱʱԴȡֵΧΪ:
  *                     QEI_TIMER_SOURCE_SCLK: ڲʱӣSCLK
  *                     QEI_TIMER_SOURCE_EXTERNAL_PIN_QEA: QEAyŵⲿʱӣ½ش
  *   ޡ
  */
void
QEI_Timer_Source_Clock_Config (QEI_SFRmap* QEIx, uint32_t TimerClock)
{
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));
    CHECK_RESTRICTION(CHECK_QEI_TIMER_SOURCE(TimerClock));

    /*---------------- QEI_CTL0ĴTXCSλ ----------------*/
    if (TimerClock != QEI_TIMER_SOURCE_SCLK)
    {
        /* QEAyŵⲿʱ */
        SFR_SET_BIT_ASM(QEIx->CTL0, QEI_CTL0_TXCS_POS);
    }
    else
    {
        /* ڲʱӣSCLK */
        SFR_CLR_BIT_ASM(QEIx->CTL0, QEI_CTL0_TXCS_POS);
    }
}

/**
  *   ȡQEI״̬
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *   1:Ϊߵƽ0:Ϊ͵ƽ
  */
FlagStatus
QEI_Get_Index_Pin_State (QEI_SFRmap* QEIx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));

    /*---------------- ȡQEI_CTL0ĴINDEXλ ----------------*/
    if (QEIx->CTL0 & QEI_CTL0_INDEX)
    {
        /* Ϊߵƽ */
        return SET;
    }
    else
    {
        /* Ϊ͵ƽ */
        return RESET;
    }
}

/**
  *   λü״̬ʹܡ
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *       NewState: λü״̬ʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
QEI_Location_Counter_Output_Enable (QEI_SFRmap* QEIx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- QEI_CTL0ĴTXDIROUTλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹλü״̬ */
        SFR_SET_BIT_ASM(QEIx->CTL0, QEI_CTL0_TXDIROUT_POS);
    }
    else
    {
        /* ֹλü״̬ */
        SFR_CLR_BIT_ASM(QEIx->CTL0, QEI_CTL0_TXDIROUT_POS);
    }
}

/**
  *   QEIλü״̬
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *       Direction: QEIλü״̬ȡֵΧΪ:
  *                    QEI_COUNT_DIRECT_REVERSE: λüΪ-
  *                    QEI_COUNT_DIRECT_FORWARD: λüΪ+
  *   ޡ
  */
void
QEI_Location_Counter_Direction_Config (QEI_SFRmap* QEIx, uint32_t Direction)
{
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));
    CHECK_RESTRICTION(CHECK_QEI_COUNT_DIRECT(Direction));

    /*---------------- QEI_CTL0ĴTXDIRλ ----------------*/
    if (Direction != QEI_COUNT_DIRECT_REVERSE)
    {
        /* λüΪ+ */
        SFR_SET_BIT_ASM(QEIx->CTL0, QEI_CTL0_TXDIR_POS);
    }
    else
    {
        /* λüΪ- */
        SFR_CLR_BIT_ASM(QEIx->CTL0, QEI_CTL0_TXDIR_POS);
    }
}

/**
  *   ȡQEIλü״̬
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *   1:λüΪ+0:λüΪ-
  */
FlagStatus
QEI_Get_Location_Counter_Direction (QEI_SFRmap* QEIx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));

    /*---------------- ȡQEI_CTL0ĴTXDIRλ ----------------*/
    if (QEIx->CTL0 & QEI_CTL0_TXDIR)
    {
        /* λüΪ+ */
        return SET;
    }
    else
    {
        /* λüΪ- */
        return RESET;
    }
}

/**
  *   õ͹ QEI ʱӡ
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *       QEICLK: ͹ QEI ʱӣȡֵΧΪ:
  *                    QEI_CLK_INTLF: ѡINTLFΪ͹QEIĹʱ
  *                    QEI_CLK_LP4M: ѡLP4MΪ͹QEIĹʱ
  *   ޡ
  */
void
QEI_LP_Source_Clock_Config(QEI_SFRmap* QEIx, uint32_t QEICLK)
{
	uint32_t tmpreg = 0;

	/* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));
    CHECK_RESTRICTION(CHECK_LPQEI_CLK(QEICLK));

    /*---------------- QEI_CTL0ĴLPCLKλ ----------------*/
    tmpreg = QEICLK << QEI_CTL0_LPCLK_POS;
    QEIx->CTL0 = SFR_Config(QEIx->CTL0, ~QEI_CTL0_LPCLK, tmpreg);
}

/**
  *   QEIʱӡ
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *       PeripheralClock: QEIʱӣȡֵΧΪ:
  *                          QEI_SOURCE_SCLK: ʱΪSCLK
  *                          QEI_SOURCE_HFCLK: ʱΪHFCLK
  *                          QEI_SOURCE_LFCLK: ʱΪLFCLK
  *   ޡ
  */
void
QEI_Source_Clock_Config (QEI_SFRmap* QEIx, uint32_t PeripheralClock)
{
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));
    CHECK_RESTRICTION(CHECK_QEI_SOURCE(PeripheralClock));

    /*---------------- QEI_CTL1ĴQEICLKλ ----------------*/
    QEIx->CTL1 = SFR_Config (QEIx->CTL1,
                            ~QEI_CTL1_QEICLK,
                            PeripheralClock);
}

/**
  *   ˲ʹܡ
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *       NewState: ˲ʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
QEI_Digital_Filter_Output_Enable (QEI_SFRmap* QEIx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- QEI_CTL1ĴQEIOUTλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹQEAy/QEBy/INDXyϵ˲ */
        SFR_SET_BIT_ASM(QEIx->CTL1, QEI_CTL1_QEIOUT_POS);
    }
    else
    {
        /* ֹ˲ */
        SFR_CLR_BIT_ASM(QEIx->CTL1, QEI_CTL1_QEIOUT_POS);
    }
}

/**
  *   QEI˲ʱӷƵ
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *       DigitalFilterClock: QEI˲ʱӷƵQEAy/QEBy/INDXy˲ʱ
  *                           зƵȡֵΧΪ:
  *                             QEI_DIGITAL_FILTER_CLK_DIV_1: 1:1Ƶ
  *                             QEI_DIGITAL_FILTER_CLK_DIV_2: 1:2Ƶ
  *                             QEI_DIGITAL_FILTER_CLK_DIV_4: 1:4Ƶ
  *                             QEI_DIGITAL_FILTER_CLK_DIV_16: 1:16Ƶ
  *                             QEI_DIGITAL_FILTER_CLK_DIV_32: 1:32Ƶ
  *                             QEI_DIGITAL_FILTER_CLK_DIV_64: 1:64Ƶ
  *                             QEI_DIGITAL_FILTER_CLK_DIV_128: 1:128Ƶ
  *                             QEI_DIGITAL_FILTER_CLK_DIV_256: 1:256Ƶ
  *   ޡ
  */
void
QEI_Digital_Filter_Clock_Config (QEI_SFRmap* QEIx, uint32_t DigitalFilterClock)
{
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));
    CHECK_RESTRICTION(CHECK_QEI_DIGITAL_FILTER_CLK(DigitalFilterClock));

    /*---------------- QEI_CTL1ĴQEICKλ ----------------*/
    QEIx->CTL1 = SFR_Config (QEIx->CTL1,
                            ~QEI_CTL1_QEICK,
                            DigitalFilterClock);
}

/**
  *   AB뽻ʹܡ
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *       NewState: AB뽻ʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
QEI_Swap_Phase_AB_Enable (QEI_SFRmap* QEIx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- QEI_CTL1ĴSWPABλ ----------------*/
    if (NewState != FALSE)
    {
        /* AB뽻 */
        SFR_SET_BIT_ASM(QEIx->CTL1, QEI_CTL1_SWPAB_POS);
    }
    else
    {
        /* AB벻 */
        SFR_CLR_BIT_ASM(QEIx->CTL1, QEI_CTL1_SWPAB_POS);
    }
}

/**
  *   QEI 16λTxʱ
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *       Counter: 16λTxʱȡֵΪ16λֵ
  *   ޡ
  */
void
QEI_Timer_Counter_Config (QEI_SFRmap* QEIx, uint16_t Counter)
{
	uint32_t tmpreg = 0;
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));

    /*---------------- Tx_CNTĴTXCNTλ ----------------*/
    tmpreg = Counter << QEI_TXCNT_TXCNT0_POS;
    QEIx->CNT = SFR_Config (QEIx->CNT,
                            ~QEI_TXCNT_TXCNT,
                            (uint32_t)tmpreg);
}

/**
  *   ȡQEI 16λTxʱ
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *   16λTxʱȡֵΪ16λֵ
  */
uint16_t
QEI_Get_Timer_Counter (QEI_SFRmap* QEIx)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));

    /*---------------- ȡTx_CNTĴTXCNTλ ----------------*/
    tmpreg = QEIx->CNT;

    return (uint16_t)tmpreg;
}

/**
  *   QEI 16λTxڼĴ
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *       Period: 16λTxڼĴȡֵΪ16λֵ
  *   ޡ
  */
void
QEI_Timer_Period_Config (QEI_SFRmap* QEIx, uint16_t Period)
{
	uint32_t tmpreg = 0;
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));

    /*---------------- Tx_PPXĴPPXλ ----------------*/
    tmpreg = Period << QEI_PPX_PPX0_POS;
    QEIx->PPX = SFR_Config (QEIx->PPX,
                            ~QEI_PPX_PPX,
                            (uint32_t)tmpreg);
}

/**
  *   ȡQEI 16λTxڼĴ
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *   16λTxڼĴȡֵΪ16λֵ
  */
uint16_t
QEI_Get_Timer_Period (QEI_SFRmap* QEIx)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));

    /*---------------- ȡTx_PPXĴPPXλ ----------------*/
    tmpreg = QEIx->PPX;

    return (uint16_t)tmpreg;
}

/**
  *   QEI 16λTxԤƵ
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *       Prescaler: 16λTxԤƵȡֵΪ16λֵ
  *   ޡ
  */
void
QEI_Timer_Prescaler_Config (QEI_SFRmap* QEIx, uint16_t Prescaler)
{
	uint32_t tmpreg = 0;
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));

    /*---------------- Tx_PRSCĴTXPRSCλ ----------------*/
    tmpreg = Prescaler << QEI_TXPRSC_TXPRSC0_POS;
    QEIx->PRSC = SFR_Config (QEIx->PRSC,
                            ~QEI_TXPRSC_TXPRSC,
                            (uint32_t)tmpreg);
}
/**
  *   ##### ·(QEI)ú #####
  */


/**
  *   ##### ·(QEI)жϹ #####
  */
/**
  *   ȡQEIжϱ־
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *       InterruptType: QEIж¼ȡֵΪṩһ:
  *                        QEI_INT_COUNT_ERROR: Txж¼
  *                        QEI_INT_PHASE_ERROR: QEIyλж¼
  *                        QEI_INT_TX_OVERFLOW: QEIyж¼
  *   1:Ӧжϱ־0:δӦжϱ־
  */
FlagStatus
QEI_Get_INT_Flag (QEI_SFRmap* QEIx, uint32_t InterruptType)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));
    CHECK_RESTRICTION(CHECK_QEI_INT_ONE_EVENT(InterruptType));

    /*---------------- ȡQEI_CTL0ĴӦλ ----------------*/
    tmpreg = InterruptType << QEI_CTL0_CNTERRIF_POS;
    if (QEIx->CTL0 & tmpreg)
    {
        /* Ӧжϱ־ */
        return SET;
    }
    else
    {
        /* δӦжϱ־ */
        return RESET;
    }
}

/**
  *   QEIжϱ־
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *       InterruptType: QEIж¼ȡֵΪһ:
  *                        QEI_INT_COUNT_ERROR: Txж¼
  *                        QEI_INT_PHASE_ERROR: QEIyλж¼
  *                        QEI_INT_TX_OVERFLOW: QEIyж¼
  *   ޡ
  */
void
QEI_Clear_INT_Flag (QEI_SFRmap* QEIx, uint32_t InterruptType)
{
    uint32_t tmpreg = 0;
    uint32_t tmpreg1 = 0;
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));
    CHECK_RESTRICTION(CHECK_QEI_INT_EVENT(InterruptType));

    /*---------------- QEI_DIERĴӦλ ----------------*/
    tmpreg1 = InterruptType << QEI_CTL0_CNTERRIF_POS;
    tmpreg = InterruptType << QEI_DIER_CNTERRIC_POS;
    QEIx->DIER |= tmpreg;
    while(QEIx->CTL0 & tmpreg1);
    QEIx->DIER &= ~tmpreg;
}

/**
  *   QEIжʹܡ
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *       InterruptType: QEIж¼ȡֵΪһ:
  *                        QEI_INT_COUNT_ERROR: Txж¼
  *                        QEI_INT_PHASE_ERROR: QEIyλж¼
  *                        QEI_INT_TX_OVERFLOW: QEIyж¼
  *       NewState: жʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
QEI_Set_INT_Enable (QEI_SFRmap* QEIx,
                    uint32_t InterruptType, FunctionalState NewState)
{
    uint32_t tmpreg = 0;

    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));
    CHECK_RESTRICTION(CHECK_QEI_INT_EVENT(InterruptType));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- QEI_DIERĴӦλ ----------------*/
    tmpreg = InterruptType << QEI_DIER_CNTERRIE_POS;
    if (NewState != FALSE)
    {
        /* ʹж */
        QEIx->DIER |= tmpreg;
    }
    else
    {
        /* ֹж */
        QEIx->DIER &= ~tmpreg;
    }
}

/**
  *   ȡTxжϱ־
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *   1:Txϻ0:Txδϻ
  */
FlagStatus
QEI_Get_Counter_Overflow_INT_Flag (QEI_SFRmap* QEIx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));

    /*---------------- ȡQEI_CTL0ĴTXIFλ ----------------*/
    if (QEIx->CTL0 & QEI_CTL0_TXIF)
    {
        /* Txϻ */
        return SET;
    }
    else
    {
        /* Txδϻ */
        return RESET;
    }
}

/**
  *   ȡQEIλжϱ־
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *   1:QEIλִ0:QEIλδִ
  */
FlagStatus
QEI_Get_Phase_Error_INT_Flag (QEI_SFRmap* QEIx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));

    /*---------------- ȡQEI_CTL0ĴPHEIFλ ----------------*/
    if (QEIx->CTL0 & QEI_CTL0_PHEIF)
    {
        /* QEIλִ */
        return SET;
    }
    else
    {
        /* QEIλδִ */
        return RESET;
    }
}

/**
  *   ȡQEIжϱ־
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *   1:QEIִ0:QEIδִ
  */
FlagStatus
QEI_Get_Count_Error_INT_Flag (QEI_SFRmap* QEIx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));

    /*---------------- ȡQEI_CTL0ĴPHEIFλ ----------------*/
    if (QEIx->CTL0 & QEI_CTL0_CNTERRIF)
    {
        /* QEIִ */
        return SET;
    }
    else
    {
        /* QEIδִ */
        return RESET;
    }
}

/**
  *   QEI Txжϱ־
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *   ޡ
  */
void
QEI_Clear_Timer_Overflow_INT_Flag (QEI_SFRmap* QEIx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));

    /*-------------------- QEI_DIERĴTXICλ --------------------*/
    SFR_SET_BIT_ASM(QEIx->DIER, QEI_DIER_TXIC_POS);
    while(QEIx->CTL0 & QEI_CTL0_TXIF);
    SFR_CLR_BIT_ASM(QEIx->DIER, QEI_DIER_TXIC_POS);
}

/**
  *   QEIλжϱ־
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *   ޡ
  */
void
QEI_Clear_Phase_Error_INT_Flag (QEI_SFRmap* QEIx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));

    /*-------------------- QEI_DIERĴPHEICλ --------------------*/
    SFR_SET_BIT_ASM(QEIx->DIER, QEI_DIER_PHEIC_POS);
    while(QEIx->CTL0 & QEI_CTL0_PHEIF);
    SFR_CLR_BIT_ASM(QEIx->DIER, QEI_DIER_PHEIC_POS);
}

/**
  *   QEIжϱ־
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *   ޡ
  */
void
QEI_Clear_Counter_Error_INT_Flag (QEI_SFRmap* QEIx)
{
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));

    /*-------------------- QEI_DIERĴCNTERRICλ --------------------*/
    SFR_SET_BIT_ASM(QEIx->DIER, QEI_DIER_CNTERRIC_POS);
    while(QEIx->CTL0 & QEI_CTL0_CNTERRIF);
    SFR_CLR_BIT_ASM(QEIx->DIER, QEI_DIER_CNTERRIC_POS);
}

/**
  *   QEI Txжʹܡ
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *       NewState: Txжʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
QEI_Timer_Overflow_INT_Enable (QEI_SFRmap* QEIx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- QEI_DIERĴTXIEλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹܼTxж */
        SFR_SET_BIT_ASM(QEIx->DIER, QEI_DIER_TXIE_POS);
    }
    else
    {
        /* ֹTxж */
        SFR_CLR_BIT_ASM(QEIx->DIER, QEI_DIER_TXIE_POS);
    }
}

/**
  *   QEIλжʹܡ
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *       NewState:λжʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
QEI_Phase_Error_INT_Enable (QEI_SFRmap* QEIx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- QEI_DIERĴPHEIEλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹQEIyλж */
        SFR_SET_BIT_ASM(QEIx->DIER, QEI_DIER_PHEIE_POS);
    }
    else
    {
        /* ֹQEIyλж */
        SFR_CLR_BIT_ASM(QEIx->DIER, QEI_DIER_PHEIE_POS);
    }
}

/**
  *   QEIжʹܡ
  *   QEIx: ָQEIڴṹָ룬ȡֵΪQEI0_SFR
  *       NewState:жʹ״̬ȡֵΪTRUE  FALSE
  *   ޡ
  */
void
QEI_Counter_Error_INT_Enable (QEI_SFRmap* QEIx, FunctionalState NewState)
{
    /* У */
    CHECK_RESTRICTION(CHECK_QEI_ALL_PERIPH(QEIx));
    CHECK_RESTRICTION(CHECK_FUNCTIONAL_STATE(NewState));

    /*---------------- QEI_DIERĴCNTERRIEλ ----------------*/
    if (NewState != FALSE)
    {
        /* ʹQEIyж */
        SFR_SET_BIT_ASM(QEIx->DIER, QEI_DIER_CNTERRIE_POS);
    }
    else
    {
        /* ֹQEIyж */
        SFR_CLR_BIT_ASM(QEIx->DIER, QEI_DIER_CNTERRIE_POS);
    }
}
/**
  *   ##### ·(QEI)жϹ #####
  */


